from __future__ import print_function

import glob
import os
import sys
import random
import importlib
import inspect
import traceback
import py_trees

from numpy import random
import carla

from agents.navigation.local_planner import RoadOption

from srunner.scenarioconfigs.scenario_configuration import ActorConfigurationData
from srunner.scenariomanager.carla_data_provider import CarlaDataProvider

from srunner.scenariomanager.scenarioatomics.atomic_behaviors import ScenarioTriggerer, Idle
from srunner.scenariomanager.scenarioatomics.atomic_trigger_conditions import WaitForBlackboardVariable
from srunner.scenariomanager.scenarioatomics.atomic_criteria import (CollisionTest,
                                                                     InRouteTest,
                                                                     RouteCompletionTest,
                                                                     OutsideRouteLanesTest,
                                                                     RunningRedLightTest,
                                                                     RunningStopTest,
                                                                     ActorBlockedTest,
                                                                     MinimumSpeedRouteTest)

from srunner.scenarios.basic_scenario import BasicScenario
# from srunner.scenarios.background_activity import BackgroundBehavior
from srunner.scenariomanager.weather_sim import RouteWeatherBehavior
from srunner.scenariomanager.lights_sim import RouteLightsBehavior
from srunner.scenariomanager.timer import RouteTimeoutBehavior

from srunner.tools.route_parser import RouteParser, DIST_THRESHOLD
from srunner.tools.route_manipulation import interpolate_trajectory

from envs.scenarios.background_activity import BackgroundBehavior
from envs.scenarios.route_scenario import RouteScenario
from envs.scenarios.multiagent_basic_scenario import MultiAgentBasicScenario
from envs.utils.parse_scenario_config import parse_scenario_config
from envs.utils.route_helper import cyclic_route_filter

SECONDS_GIVEN_PER_METERS = 0.4

class MultiAgentScenario(MultiAgentBasicScenario):
    """
    Implements a multi-agent scenario that prepares agents and scenarios
    Assuming a single scenario is triggered
    """
    category = "MultiAgentScenario"

    def __init__(self,
                 world: carla.World,
                 config,
                 debug_mode: int = 0,
                 terminate_on_failure: bool = True,
                 criteria_enable: bool = True
                 ) -> None:
        """
        Setup all relevant parameters and create scenario
        """
        self.client = CarlaDataProvider.get_client()
        self.scenario_config = config
        self.carla_scenario_config = parse_scenario_config(config)
        self.world = world
        self.map = CarlaDataProvider.get_map()
        self.timeout = 10000
        self.list_scenarios = []

        self.behavior_node = None
        self.criteria_node = None

        self.ego_vehicles = self.spawn_ego_vehicles()
        if not self.ego_vehicles:
            raise Exception("Error: Unable to spawn vehicles")
        self.routes = self.get_routes()

        if debug_mode > 0:
            self._debug_mode = True
            #self._draw_waypoints()
        
        self.terminate_on_failure = terminate_on_failure
        self.criteria_enable = criteria_enable
        super().__init__(
            config.scenarioname, self.ego_vehicles, self.carla_scenario_config,
            world, debug_mode=debug_mode, terminate_on_failure=terminate_on_failure,
            criteria_enable=criteria_enable)

        # Do it after the 'super', as we need the behavior and criteria tree initialized
        self.build_scenario(debug=debug_mode)
        print(CarlaDataProvider._carla_actor_pool.keys())
        # Set runtime init mode. Do this after the first set of scenarios has been built
        CarlaDataProvider.set_runtime_init_mode(True)

    def spawn_ego_vehicles(self):
        """
        spawn the controllable vehicles specified in config
        """
        vehicles = []
        for vehicle in self.scenario_config.vehicles:
            vehicle_location = carla.Location()
            vehicle_location.x = vehicle.start.x
            vehicle_location.y = vehicle.start.y
            vehicle_location.z = vehicle.start.z + 0.5
            vehicle_waypoint = self.map.get_waypoint(vehicle_location)
            vehicle_transform = vehicle_waypoint.transform
            if 'random' in vehicle.model.lower():
                # Randomly select a model if 'random' is specified in config
                models = self.world.get_blueprint_library().filter('vehicle.*.*')
                vehicle.model = random.choice(models).id
            vehicle_transform.rotation.roll = vehicle.start.get('roll', vehicle_transform.rotation.roll)
            vehicle_transform.rotation.pitch = vehicle.start.get('pitch', vehicle_transform.rotation.pitch)
            vehicle_transform.rotation.yaw = vehicle.start.get('yaw', vehicle_transform.rotation.yaw)
            vehicles.append(CarlaDataProvider.request_new_actor(
                            model=vehicle.model,
                            spawn_point=vehicle_transform,
                            rolename=vehicle.rolename,
                            ))
        self.world.tick()
        return vehicles

    def get_routes(self):
        routes = []
        for vehicle in self.scenario_config.vehicles:
            keypoints = []
            for wp in vehicle.route:
                loc = carla.Location(x=wp.x, y=wp.y, z=wp.z)
                keypoints.append(loc)
            gps_route, route = interpolate_trajectory(keypoints)
            route = cyclic_route_filter(route)
            routes.append(route)
        return routes

    def _initialize_actors(self, config) -> None:
        """
        Spawn the other actors specified in config
        """
        print("config.other_actors: ", config.other_actors)

    def get_all_scenario_classes(self):
        """
        Search through the 'multiagent_scenarios' folder for all the Python classes
        """
        scenarios_list = glob.glob('envs/scenarios/multiagent_scenarios/*/*.py')
        scenario_classes = {}
        for scenario_file in scenarios_list:
            module_name = os.path.basename(scenario_file).split('.')[0]
            sys.path.insert(0, os.path.dirname(scenario_file))
            scenario_module = importlib.import_module(module_name)
            for member in inspect.getmembers(scenario_module, inspect.isclass):
                scenario_classes[member[0]] = member[1]
        return scenario_classes

    def build_scenario(self, debug=False):
        """
        Initializes the class of all the scenarios that will be present in the config
        If a class failed to be initialized, a warning is printed
        """
        new_scenarios = []
        self.scenario_classes = self.get_all_scenario_classes()
        # Part 1. Check all scenarios that haven't been initialized yet
        scenario_class = self.scenario_classes[self.scenario_config.scenarioname]
        scenario_instance = scenario_class(self.world,
                                           self.ego_vehicles,
                                           self.carla_scenario_config,
                                           timeout=self.timeout)
        new_scenarios.append(scenario_instance)
        self.list_scenarios.append(scenario_instance)
        # Part 2. Add their behavior onto the root behavior tree
        for scenario in new_scenarios:
            # Add the behavior
            if scenario.behavior_tree is not None:
                self.behavior_node.add_child(scenario.behavior_tree)
            # Add the criteria
            scenario_criteria = scenario.get_criteria()
            if scenario_criteria:
                self.criteria_node.add_child(
                    self._create_criterion_tree(scenario, scenario_criteria)
                )
            self.other_actors.extend(scenario.other_actors)

    def _create_weather_behavior(self):
        """
        Create the weather behavior
        """
        # Set the appropriate weather conditions
        weather = self.config.weather
        self.world.set_weather(weather)
        return None
        #return RouteWeatherBehavior(self.ego_vehicles[0], self.route, self.config.weather)
    
    def _create_behavior(self):
        """
        Creates a parallel behavior that runs all of the scenarios part of the route.
        These subbehaviors have had a trigger condition added so that they wait until
        the agent is close to their trigger point before activating.

        It also adds the BackgroundActivity scenario, which will be active throughout the whole route.
        This behavior never ends and the end condition is given by the RouteCompletionTest criterion.
        """
        scenario_trigger_distance = DIST_THRESHOLD  # Max trigger distance between route and scenario

        behavior = py_trees.composites.Parallel(name="Behavior",
                                                policy=py_trees.common.ParallelPolicy.SUCCESS_ON_ALL)

        self.behavior_node = behavior
        scenario_behaviors = []
        blackboard_list = []

        # Add the Background Activity
        # TODO: buggy right now as it builds on the ego vehicle
        # behavior.add_child(BackgroundBehavior(self.ego_vehicles[0],
        #                                       self.routes[0],
        #                                       name="BackgroundActivity",
        #                                       traffic_density=0.0,
        #                                       ))

        # Add the behavior that manages the scenario trigger conditions
        # scenario_triggerer = ScenarioTriggerer(
        #     self.ego_vehicles[0], self.routes[0], blackboard_list, scenario_trigger_distance)
        # behavior.add_child(scenario_triggerer)  # Tick the ScenarioTriggerer before the scenarios

        behavior.add_children(scenario_behaviors)
        return behavior

    def _create_test_criteria(self):
        """
        This is called when initialization of the class.
        Create the criteria tree. It starts with some route criteria (which are always active),
        and adds the scenario specific ones, which will only be active during their scenario
        """
        criteria = py_trees.composites.Parallel(name="Criteria",
                                                policy=py_trees.common.ParallelPolicy.SUCCESS_ON_ALL)

        self.criteria_node = criteria
        for i, vehicle in enumerate(self.ego_vehicles):
            # Route completion criteria
            criteria.add_child(RouteCompletionTest(vehicle, route=self.routes[i]))
            # Collision detection
            criteria.add_child(CollisionTest(vehicle, name="CollisionTest", terminate_on_failure=True))
            # criteria.add_child(RunningRedLightTest(vehicle)) TODO: removed because it causes bugs
            # criteria.add_child(RunningStopTest(vehicle)) TODO: removed because it causes bugs

        return criteria

    def _create_criterion_tree(self, scenario, criteria):
        """
        We can make use of the blackboard variables used by behaviors themselves,
        as we already have an atomic that handles their (de)activation.
        The criteria will wait until that variable is active (the scenario has started),
        and will automatically stop when it deactivates (as the scenario has finished).
        """
        scenario_name = scenario.name
        criteria_tree = py_trees.composites.Sequence(name="scenario_name")
        scenario_criteria = py_trees.composites.Parallel(name=scenario_name,
                                                         policy=py_trees.common.ParallelPolicy.SUCCESS_ON_ALL)
        for criterion in criteria:
            scenario_criteria.add_child(criterion)
        criteria_tree.add_child(scenario_criteria)
        criteria_tree.add_child(Idle()) # Avoid the individual criteria stopping the simulation
        return criteria_tree
